#!/usr/bin/env bash
#
# decode_pad.sh – Reverse of build_pad.sh (OTP decryption)
#
# Usage:
#   ./decode_pad.sh <line_file> <pgp_wrapped_message> <output_plaintext>
#
#   <line_file>            – one line of raw symbols (cards, dice, coins)
#   <pgp_wrapped_message> – file that contains:
#                               -----BEGIN PGP MESSAGE-----
#                               <Base64 ciphertext>
#                               -----END PGP MESSAGE-----
#   <output_plaintext>    – recovered UTF‑8 plaintext will be written here
#
# The script accepts any of the card spellings defined in
# scripts/utils/card_map.txt (Unicode glyphs, two‑letter codes, lower‑case,
# alternate “T10C” style, etc.).  It normalises tokens in the same way as
# build_pad.sh, so the same line can be used for encryption and decryption
# without any manual conversion.

set -euo pipefail

# ----------------------------------------------------------------------
# Helper: print usage and exit
# ----------------------------------------------------------------------
usage() {
    echo "Usage: $0 <line_file> <pgp_wrapped_message> <output_plaintext>"
    exit 1
}
[[ $# -eq 3 ]] || usage

LINE_FILE="$1"
PGP_MSG_FILE="$2"
OUT_PLAIN="$3"

# ----------------------------------------------------------------------
# Locate this script's directory and the utils folder
# ----------------------------------------------------------------------
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
UTILS_DIR="${SCRIPT_DIR}/utils"

# ----------------------------------------------------------------------
# Load mapping tables (card, die, coin) into associative arrays
# ----------------------------------------------------------------------
declare -A CARD_MAP DIE_MAP COIN_MAP

# --------- Card map ----------
while read -r token _; do
    [[ -z "$token" || "$token" == \#* ]] && continue
    norm=$(echo "$token" | tr '[:lower:]' '[:upper:]' \
                       | sed -e 's/♣/C/g' -e 's/♦/D/g' -e 's/♥/H/g' -e 's/♠/S/g')
    CARD_MAP["$norm"]=1
done < "${UTILS_DIR}/card_map.txt"

# --------- Die map ----------
while read -r token value; do
    [[ -z "$token" ]] && continue
    DIE_MAP["$token"]=$value
done < "${UTILS_DIR}/die_map.txt"

# --------- Coin map ----------
while read -r token value; do
    [[ -z "$token" ]] && continue
    COIN_MAP["$token"]=$value
done < "${UTILS_DIR}/coin_map.txt"

# ----------------------------------------------------------------------
# Extract the Base64 payload from the fake PGP block
# ----------------------------------------------------------------------
b64_payload=$(awk '
    $0 ~ /^-----BEGIN PGP MESSAGE-----$/ {found=1; next}
    $0 ~ /^-----END PGP MESSAGE-----$/   {found=0}
    found {print}
' "$PGP_MSG_FILE")

if [[ -z "$b64_payload" ]]; then
    echo "ERROR: Could not find a Base64 payload between PGP delimiters in $PGP_MSG_FILE." >&2
    exit 1
fi

# Decode Base64 to a temporary binary ciphertext file
cipher_tmp=$(mktemp)
printf "%s" "$b64_payload" | base64 -d > "$cipher_tmp"

cipher_len=$(stat -c%s "$cipher_tmp")
if (( cipher_len == 0 )); then
    echo "ERROR: Decoded ciphertext is empty." >&2
    rm -f "$cipher_tmp"
    exit 1
fi

# ----------------------------------------------------------------------
# Convert the raw symbols line into a binary keystream (same as build_pad.sh)
# ----------------------------------------------------------------------
bits=""

lineno=0
while read -r raw_token; do
    ((lineno++))
    token=$(echo "$raw_token" | tr '[:lower:]' '[:upper:]' \
                               | sed -e 's/♣/C/g' -e 's/♦/D/g' -e 's/♥/H/g' -e 's/♠/S/g')
    if [[ -n "${CARD_MAP[$token]+x}" ]]; then
        # Find numeric value for this token (first column in card_map.txt)
        num=$(grep -E "^[[:space:]]*$raw_token[[:space:]]" "${UTILS_DIR}/card_map.txt" \
               | awk '{print $1}' | head -n1)
        if [[ -z "$num" ]]; then
            echo "ERROR: Could not locate numeric value for card token \"$raw_token\" (line $lineno)." >&2
            rm -f "$cipher_tmp"
            exit 1
        fi
        bits+=$(printf "%06d" "$(bc <<< "obase=2;$((num))")")
    elif [[ -n "${DIE_MAP[$token]+x}" ]]; then
        num=${DIE_MAP[$token]}
        bits+=$(printf "%03d" "$(bc <<< "obase=2;$((num))")")
    elif [[ -n "${COIN_MAP[$token]+x}" ]]; then
        bits+="${COIN_MAP[$token]}"
    else
        echo "ERROR: Unknown token \"$raw_token\" (line $lineno) in $LINE_FILE." >&2
        rm -f "$cipher_tmp"
        exit 1
    fi
done < <(tr -s '[:space:]' '\n' < "$LINE_FILE")

# Pad to a whole number of bytes (add trailing zeros if needed)
rem=$(( ${#bits} % 8 ))
if (( rem != 0 )); then
    pad=$((8 - rem))
    bits+=$(printf "%0${pad}d" 0)
fi

# Write bits to a temporary keystream file
keystream_tmp=$(mktemp)
printf "%s" "$bits" | xxd -r -b > "$keystream_tmp"

keystream_len=$(stat -c%s "$keystream_tmp")
if (( keystream_len < cipher_len )); then
    echo "ERROR: Keystream ($keystream_len bytes) shorter than ciphertext ($cipher_len bytes)." >&2
    rm -f "$cipher_tmp" "$keystream_tmp"
    exit 2
fi

# Truncate keystream to exactly the ciphertext length
keystream_trunc=$(mktemp)
dd if="$keystream_tmp" of="$keystream_trunc" bs=1 count="$cipher_len" status=none
rm -f "$keystream_tmp"

# ----------------------------------------------------------------------
# XOR ciphertext with keystream → recovered plaintext
# ----------------------------------------------------------------------
plain_tmp=$(mktemp)

perl -e '
    open(my $ct, "<", $ARGV[0]) or die "cannot open ciphertext: $!";
    open(my $ks, "<", $ARGV[1]) or die "cannot open keystream: $!";
    open(my $out, ">", $ARGV[2]) or die "cannot write plaintext: $!";
    while (read($ct, my $cb, 1) && read($ks, my $kb, 1)) {
        print $out chr(ord($cb) ^ ord($kb));
    }
' "$cipher_tmp" "$keystream_trunc" "$plain_tmp"

mv "$plain_tmp" "$OUT_PLAIN"
rm -f "$cipher_tmp" "$keystream_trunc"

# ----------------------------------------------------------------------
# Summary output
# ----------------------------------------------------------------------
echo "[INFO] Ciphertext length : $cipher_len bytes"
echo "[INFO] Keystream length  : $keystream_len bytes (truncated to $cipher_len for XOR)"
echo "[INFO] Plaintext written to $OUT_PLAIN"
exit 0

How to use the script (mirrors the encryption flow)

# 1️⃣  Export the same OTP line you used for encryption
echo "K♠ ks KC 10c AD t h H T 3 5 2" > recv_line.txt

# 2️⃣  Decrypt the PGP‑wrapped message you received
../scripts/decode_pad.sh recv_line.txt real_msg.txt recovered.txt

# 3️⃣  View the recovered plaintext
cat recovered.txt